% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rhinotypeR.R
\docType{package}
\name{rhinotypeR}
\alias{rhinotypeR}
\alias{rhinotypeR-package}
\title{rhinotypeR: VP4/2-based genotyping and visualization for human rhinoviruses}
\description{
The \pkg{rhinotypeR} package provides a toolkit for automating rhinovirus
genotyping using the VP4/2 genomic region—streamlining alignment, genotype
assignment, and visualization.
}
\section{Main Functions}{

\subsection{Core Analysis Functions:}{
\itemize{
  \item \code{\link{assignTypes}}: Assign sequences to rhinovirus genotypes
        based on distances to prototype strains. Returns query ID, assigned type,
        distance to nearest prototype, and reference used.
  \item \code{\link{pairwiseDistances}}: Calculates pairwise genetic distances
        between all sequences using specified evolutionary models (e.g., IUPAC).
  \item \code{\link{overallMeanDistance}}: Computes the overall mean genetic
        distance across all sequences as a measure of diversity.
}}

\subsection{Alignment and Data Preparation:}{
\itemize{
  \item \code{\link{alignToRefs}}: Aligns user sequences with packaged prototype
        references using ClustalW, ClustalOmega, or MUSCLE. Optionally crops
        the alignment to a reference span.
  \item \code{\link{getPrototypeSeqs}}: Exports packaged prototype sequences
        to a local directory for external alignment workflows.
  \item \code{deleteMissingDataSites}: Performs global gap deletion
        (removes any column containing gaps in any sequence).  % internal helper, not linked
}}

\subsection{Visualization Functions:}{
\itemize{
  \item \code{\link{plotDistances}}: Creates a heatmap of pairwise genetic distances.
  \item \code{\link{plotTree}}: Generates a simple phylogenetic tree from a distance matrix.
  \item \code{\link{SNPeek}}: Visualizes single nucleotide polymorphisms (SNPs)
        with color-coded nucleotides (A=green, T=red, C=blue, G=yellow).
  \item \code{\link{plotAA}}: Displays amino acid differences with zooming and highlighting.
  % If present:
  % \item \code{\link{plotFrequency}}: Visualizes the frequency distribution of assigned genotypes.
}}
}

\section{Input expectations}{

\itemize{
  \item \emph{Format:} FASTA; DNA for VP4/2 (AA only for \code{\link{plotAA}}).
  \item \emph{Region:} VP4/2; sequences must be homologous.
  \item \emph{Length/QC:} Recommended \eqn{\ge} 350 bp (typical 420 bp) and trimmed to the VP4/2 span.
}
}

\section{Workflows}{

\subsection{Workflow A: External Alignment}{
\enumerate{
  \item Export prototypes: \code{getPrototypeSeqs("~/Desktop")}
  \item Combine prototypes with your sequences
  \item Align externally (e.g., MAFFT \url{https://mafft.cbrc.jp/alignment/server/})
  \item Import alignment: \code{Biostrings::readDNAStringSet("alignment.fasta")}
  \item Assign genotypes: \code{assignTypes(alignment)}
}}

\subsection{Workflow B: Align in R}{
\preformatted{
# Align sequences with packaged prototypes
aln <- alignToRefs(
  seqData   = rhinovirusVP4,
  method    = "ClustalW",
  trimToRef = TRUE,
  refName   = "JN855971.1_A107"
)

# Assign genotypes
res <- assignTypes(
  aln,
  model = "IUPAC",
  deleteGapsGlobally = FALSE,
  threshold = 0.105
)

# Visualize results
plotFrequency(res)
}
}
}

\section{Key Parameters}{

\itemize{
  \item \strong{threshold:} Default 0.105 (approximately 10.5\%) for VP4/2 region, following
        proposals in McIntyre et al. (2013).
  \item \strong{trimToRef:} When TRUE, crops alignment to non-gap span of chosen
        prototype (helpful when reads extend beyond target region).
  \item \strong{deleteGapsGlobally:} Global deletion vs. pairwise deletion (distance models).
  \item \strong{model:} Evolutionary model for distance calculation (e.g., "IUPAC").
}
}

\section{Bundled datasets}{

\itemize{
  \item \code{rhinovirusPrototypesVP4}: VP4/2 prototype references as a
        \code{Biostrings::DNAStringSet} (mirrors \file{inst/extdata/prototypes.fasta}).
        Reported by McIntyre et al. (2013), \emph{Journal of General Virology}, 94(8), 1791–1806.
        \doi{10.1099/vir.0.053686-0}
  \item \code{rhinovirusVP4}: A rhinovirus VP4/2 alignment as a
        \code{Biostrings::DNAStringSet} for examples/tests; derived from
        Luka et al. (2020), \emph{Open Forum Infectious Diseases}, 7(10), ofaa385.
        \doi{10.1093/ofid/ofaa385}
}
}

\section{Getting Started}{

\preformatted{
library(rhinotypeR)
data(rhinovirusVP4, package = "rhinotypeR")

aln <- alignToRefs(
  seqData   = rhinovirusVP4,
  method    = "ClustalW",
  trimToRef = TRUE
)
res <- assignTypes(aln, model = "IUPAC")
head(res)

d <- pairwiseDistances(rhinovirusVP4, model = "IUPAC")
plotDistances(d)

sampled <- d[1:30, 1:30]
plotTree(sampled, main = "Rhinovirus VP4/2 Tree")

SNPeek(rhinovirusVP4)
aa <- Biostrings::translate(rhinovirusVP4)
plotAA(aa)
}
}

\section{Troubleshooting}{

\itemize{
  \item \strong{Prototypes missing error:} Ensure prototypes are present. Use
        \code{alignToRefs()} or \code{getPrototypeSeqs()}.
  \item \strong{Misaligned regions:} Use \code{trimToRef = TRUE} and choose an
        appropriate \code{refName}.
  \item \strong{Gap issues:} Consider global deletion with \code{deleteMissingDataSites()}
        or pairwise deletion via distance models.
  \item \strong{Truncated plots:} Increase device height or use
        \code{show_only_highlighted = TRUE}.
}
}

\seealso{
\code{\link{alignToRefs}}, \code{\link{assignTypes}}, \code{\link{pairwiseDistances}},
\code{\link{overallMeanDistance}}, \code{\link{plotDistances}},
\code{\link{plotTree}}, \code{\link{SNPeek}}, \code{\link{plotAA}},
\code{\link{getPrototypeSeqs}}
}
\author{
\strong{Maintainer}: Martha Luka \email{marthaluka20@gmail.com} (\href{https://orcid.org/0000-0001-6217-4426}{ORCID})

Authors:
\itemize{
  \item Ruth Nanjala
  \item Winfred Gatua
  \item Wafaa M. Rashed
  \item Olaitan Awe
}

}
\keyword{package}
